/*
 * File........: c:/ARC/PROG/java/gk/app/speed/SpeedTimer.java
 * Package.....: gk.app.speed
 * Created.....: 98/09/03, Guido Krueger
 * RCS.........: $Revision: 1.1 $
 *               $Date: 1998/10/25 12:49:28 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.app.speed;

import gk.util.*;

/**
 * Class for timing speed critical code. Timing is done in three steps:
 * <ul>
 * <li>Measure the hardware timer resolution using the 
 *     checkResolution method.
 * <li>Measure the overhead of empty loop iteration using
 *     the calibrateLoop method.
 * <li>Measure the speed of the code by successively running it in 
 *     a loop until the elapsed time is long enough to give meaningful
 *     results.
 * </ul>
 */
public class SpeedTimer
{
  //---Pseudo Constants------------------------------------------
  protected static int TIMEPERTEST;  //min time per test

  //---Instance variables----------------------------------------
  protected double resolution; //hardware timer resolution in milliseconds
  protected double looptime;   //time needed for one iteration of for loop [ms]

  /**
   * Creates a new SpeedTimer object.
   */
  public SpeedTimer()
  {
	resolution = 0.0;
	looptime = 0.0;
  }

  /**
   * Computes the hardware timer resolution of the system the
   * program is running on. The value can be retrieved using 
   * the getResolution method.
   */
  public void checkResolution()
  {
	long t1, t2;
	int  sumres = 0;
	System.gc();
	for (int i = 1; i <= 20; ++i) {
	  t1 = System.currentTimeMillis();
	  while (true) {
		t2 = System.currentTimeMillis();
		if (t2 != t1) {
		  sumres += (int)(t2 - t1);
		  break;
		}
	  }
	}
	resolution = sumres / (double)20;
	TIMEPERTEST = (int)(50 * resolution);
  }

  /**
   * Computes the time which is needed to run a single iteration of an 
   * empty loop. The result can be retrieved using the getLoopTime method.
   */
  public void calibrateLoop()
  {
	long cnt, loops = 0, t1, t2, runtime;
	System.gc();
	do {
	  //double number of iterations
	  loops = 2 * loops + 1;
	  t1 = System.currentTimeMillis();
	  for (cnt = 0; cnt < loops; ++cnt) {
		//empty for calibration purposes
	  }
	  t2 = System.currentTimeMillis();
	  runtime = t2 - t1;
	} while (runtime < TIMEPERTEST);
	looptime = (double)runtime / cnt;
  }

  /**
   * Returns the hardware timer resolution. Should be called only after
   * a successful call to checkResolution().
   */
  public double getResolution()
  {
	return resolution;
  }

  /**
   * Returns the minimum time used to run each test. The value is
   * computed in checkResolution() as 50 * resolution.
   */
  public double getTimePerTest()
  {
	return TIMEPERTEST;
  }

  /**
   * Returns the time needed to run a single iteration of an empty loop.
   * Should be called only after a successful call to calibrateLoop().
   */
  public double getLoopTime()
  {
	return looptime;
  }

  /**
   * Measures the time which is needed to run a particular statement
   * sequence. The return value is in milliseconds and is retrieved by
   * successively calling the statements in a growing loop until the 
   * elapsed time (retrieved by System.currentTimeMillis()) is long
   * enough to give meaningful results.
   */
  public double timeStatements()
  {
	long cnt, loops = 0, t1, t2, runtime;
	System.gc();
	do {
	  //double the number of iterations
	  loops = 2 * loops + 1;
	  //System.out.print("loops = " + loops);
	  t1 = System.currentTimeMillis();
	  for (cnt = 0; cnt < loops; ++cnt) {
		//--->>>statements to be timed should start here<<<---
		Object o = new Object();
		//--->>>end of statements to be timed<<<---
	  }
	  t2 = System.currentTimeMillis();
	  runtime = t2 - t1;
	  //System.out.println(": " + runtime);
	} while (runtime < TIMEPERTEST);
	return ((double)runtime / cnt) - looptime;
  }

  //---Static methods---------------------------------------------
  /**
   * Converts the given time into a 10^-3*n fractional unit of a second
   * and returns both value and unit as a String.
   */
  public static String getTimeFormatted(double millis)
  {
	String aUnits[] = {"s.", "ms.", "us.", "ns.", "ps."};
	int nUnit = 0;
	//Convert to non-negative seconds
	int sign = millis < 0 ? -1 : 1;
	millis = Math.abs(millis / 1000.0);
	while (true) {
	  if (millis >= 1.0 || nUnit >= aUnits.length - 1) {
		break;
	  }
	  millis *= 1000.0;
	  ++nUnit;
	}
	return Str.getFormatted("%.1f ", sign * millis) + aUnits[nUnit];
  }

  //---Testing only---------------------------------------------
  public static void main(String args[])
  {
	SpeedTimer st = new SpeedTimer();
	//check resolution
	st.checkResolution();
	System.out.println("resolution is " + st.getResolution() + " ms.");
	System.out.println("time per test is " + st.getTimePerTest() + " ms.");
	//calibrate
	st.calibrateLoop();
	System.out.println(
	  "calibrated loop overhead is " +
	  SpeedTimer.getTimeFormatted(st.getLoopTime())
	);
	//time test code
	System.out.println(
	  "timed statements are " +
	  SpeedTimer.getTimeFormatted(st.timeStatements())
	);
  }
}
