/*
 * File........: c:/ARC/PROG/java/gk/app/speed/SpeedTimerGenerator.java
 * Package.....: gk.app.speed
 * Created.....: 98/09/03, Guido Krueger
 * RCS.........: $Revision: 1.1 $
 *               $Date: 1998/10/25 12:49:19 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.app.speed;

import java.io.*;
import java.util.*;

/**
 * Creates a new SpeedTimerClient class which is derived from
 * the SpeedTimer class. The contents of the class is based
 * on a textual description with must have the following format:
 * <pre>
 * codeline
 * codeline
 * ...
 * ---
 * codeline
 * codeline
 * ...
 * ---
 * ...
 * </pre>
 * For each codeline sequence, the generator creates a method
 * with the given code running inside a timing loop. In the 
 * main method, it also creates code to call this method and 
 * to write the timing results to standard output. The 
 * description file's name is given as an argument to the
 * SpeedTimerGenerator file.
 */
public class SpeedTimerGenerator
{
  //---Pseudo constants-----------------------------------
  static final String NL = System.getProperty("line.separator");
  static final String OUTCLASS = "SpeedTimerClient";

  //---Instance variables---------------------------------
  Vector codeblocks1; //code for source inclusion
  Vector codeblocks2; //code for output generation
  PrintWriter out;

  /**
   * Reads the input file and stores each codeblock in the codeblocks
   * Vector.
   */  
  public void parseInputFile(String filename)
  throws IOException
  {
	codeblocks1 = new Vector();
	codeblocks2 = new Vector();
	StringBuffer srclines1 = new StringBuffer(200);
	StringBuffer srclines2 = new StringBuffer(200);
	String line;
	BufferedReader in = new BufferedReader(
						new FileReader(filename));
	while (true) {
	  if ((line = in.readLine()) == null) {
		break;
	  }
	  if (line.startsWith("---")) {
		codeblocks1.addElement(srclines1.toString());
		codeblocks2.addElement(srclines2.toString());
		srclines1.setLength(0);
		srclines2.setLength(0);
	  } else {
		srclines1.append("        " + line + NL);
		srclines2.append(escape(line) + "\\n");
	  }
	}
	if (srclines1.length() > 0) {
	  codeblocks1.addElement(srclines1.toString());
	  codeblocks2.addElement(srclines2.toString());
	}
	in.close();
  }

  /**
   * Creates the output file and inserts the header, timer and
   * trailer sections.
   */
  public void createOutputFile()
  throws IOException
  {
	out = new PrintWriter(
          new BufferedWriter(
          new FileWriter(OUTCLASS + ".java")));
	createHeader();
	createTimerSections();
	createTrailer();
	out.close();
  }

  //---Private methods----------------------------------------
  /**
   * Creates the header section of the SpeedTimerClient class.
   */
  private void createHeader()
  {
	out.println("/* " + OUTCLASS + ".java");
	out.println("   Do not modify - Code was generated by SpeedTimeGenerator");
	out.println("*/");
	out.println("");
	out.println("import gk.app.speed.*;");
	out.println("");
	out.println("public class " + OUTCLASS);
	out.println("extends SpeedTimer");
	out.println("{");
  }

  /**
   * Creates a method with a timing loop for each of the code
   * sections given in the description file.
   */
  private void createTimerSections()
  {
	int cnt = 0;
	Enumeration e = codeblocks1.elements();
	while (e.hasMoreElements()) {
	  String source = (String) e.nextElement();
	  ++cnt;
	  out.println("");
	  out.println("  public double timeStatements" + cnt + "()");
	  out.println("  {");
	  out.println("     long cnt, loops = 0, t1, t2, runtime;");
	  out.println("     System.gc();");
	  out.println("     do {");
      out.println("      //double the number of iterations");
      out.println("      loops = 2 * loops + 1;");
      out.println("      //System.out.print(\"loops = \" + loops);");
      out.println("      t1 = System.currentTimeMillis();");
      out.println("      for (cnt = 0; cnt < loops; ++cnt) {");
	  out.println("        //--->>>start of statements to be timed<<<---");
	  out.println(source);
	  out.println("        //--->>>end of statements to be timed<<<---");
	  out.println("      }");
	  out.println("      t2 = System.currentTimeMillis();");
	  out.println("      runtime = t2 - t1;");
	  out.println("      //System.out.println(\": \" + runtime);");
	  out.println("    } while (runtime < TIMEPERTEST);");
	  out.println("    return ((double)runtime / cnt) - looptime;");
	  out.println("  }");
	}
  }

  /**
   * Creates the main method for the SpeedTimerClient class.
   */
  private void createTrailer()
  {
	out.println("");
    out.println("  public static void main(String args[])");
	out.println("  {");
    out.println("    SpeedTimerClient stc = new SpeedTimerClient();");
	out.println("    stc.checkResolution();");
	out.println("    System.out.println(\"resolution is \" + stc.getResolution() + \" ms.\");");
	out.println("    System.out.println(\"time per test is \" + stc.getTimePerTest() + \" ms.\");");
	out.println("    stc.calibrateLoop();");
	out.println("    System.out.println(");
	out.println("      \"calibrated loop overhead is \" +");
	out.println("      SpeedTimer.getTimeFormatted(stc.getLoopTime())");
	out.println("    );");

	out.println("");
	out.println("    double time;");

	int cnt = 0;
	Enumeration e = codeblocks2.elements();
	while (e.hasMoreElements()) {
	  String source = (String) e.nextElement();
	  ++cnt;
	  out.println("");
	  out.println("    System.out.println(\"-----------\");");
	  out.println("");
	  out.println("    System.out.print(\"" + source + "\");");
	  out.println("    time = stc.timeStatements" + cnt + "();");
	  out.println("    System.out.println(");
      out.println("      \"(\" + SpeedTimer.getTimeFormatted(time) + \")\"");
      out.println("    );");
	}

	out.println("  }");
	out.println("}");
  }

  /**
   * Escapes the escape chars of a given code section.
   */
  private String escape(String src)
  {
	int len = src.length();
	StringBuffer ret = new StringBuffer(len + 100);
	for (int i = 0; i < len; ++i) {
	  char c = src.charAt(i);
	  switch (c) {
	  case '\"':
		ret.append("\\\"");
		break;
	  case '\'':
		ret.append("\\\'");
		break;
	  case '\\':
		ret.append("\\\\");
		break;
	  default:
		ret.append(c);
		break;
	  }
	}
	return ret.toString();
  }

  //---main method----------------------------------------
  public static void main(String args[])
  {
	if (args.length != 1) {
	  System.err.println("Usage: java SpeedTimeGenerator <inputfile>");
	  System.exit(1);
	}
	SpeedTimerGenerator stg = new SpeedTimerGenerator();
	try {
	  stg.parseInputFile(args[0]);
	  stg.createOutputFile();
	} catch (IOException e) {
	  System.err.println(e.toString());
	  System.exit(1);
	}
  }
}
