/*
 * File........: c:/ARC/PROG/java/gk/lscript/CorePlugin.java
 * Package.....: gk.lscript
 * Created.....: 98/08/09, Guido Krueger
 * RCS.........: $Revision: 1.4 $
 *               $Date: 1998/08/15 01:17:36 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.lscript;

public class CorePlugin
extends LScriptPlugin
{
  //Function numbers constants----------------------------------------
  private final int FUNC_EVAL      =  1;
  private final int FUNC_IDENTITY  =  2;
  private final int FUNC_PROGN     =  3;
  private final int FUNC_COND      =  4;
  private final int FUNC_WHILE     =  5;
  private final int FUNC_PRINT     =  6;
  private final int FUNC_PRINTLN   =  7;
  private final int FUNC_QUIT      =  8;
  private final int FUNC_BREAK     =  9;

  //Instance variables----------------------------------------
  protected Eval eval;

  public CorePlugin()
  {
  }

  public void init(Eval eval)
  throws LScriptException
  {
	this.eval = eval;
	//Functions
	eval.registerFunction("eval",     this, FUNC_EVAL);
	eval.registerFunction("identity", this, FUNC_IDENTITY);
	eval.registerFunction("progn",    this, FUNC_PROGN);
	eval.registerFunction("cond",     this, FUNC_COND);
	eval.registerFunction("while",    this, FUNC_WHILE);
	eval.registerFunction("print",    this, FUNC_PRINT);
	eval.registerFunction("println",  this, FUNC_PRINTLN);
	eval.registerFunction("quit",     this, FUNC_QUIT);
	eval.registerFunction("break",    this, FUNC_BREAK);
  }

  public Term execute(int funcnum, Term paras, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	switch (funcnum) {
	case FUNC_EVAL:
	  ret = eval(paras, paracnt);
	  break;
	case FUNC_IDENTITY:
	  ret = identity(paras, paracnt);
	  break;
	case FUNC_PROGN:
	  ret = progn(paras, paracnt);
	  break;
	case FUNC_COND:
	  ret = cond(paras, paracnt);
	  break;
	case FUNC_WHILE:
	  ret = whilefunc(paras, paracnt);
	  break;
	case FUNC_PRINT:
	  ret = print(paras, paracnt, false);
	  break;
	case FUNC_PRINTLN:
	  ret = print(paras, paracnt, true);
	  break;
	case FUNC_BREAK:
	  ret = quitbreak(paras, paracnt, LScriptException.BREAK);
	  break;
	case FUNC_QUIT:
	  ret = quitbreak(paras, paracnt, LScriptException.QUIT);
	  break;
	default:
	  throw new LScriptException(
        "internal error: unknown function number in CorePlugin"
	  );
	}
	return ret;
  }

  public void destroy()
  {
  }

  //Function implementations----------------------------------------
  private Term eval(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("?", args, paracnt, false, 1);
	return eval.eval(paras[0]);
  }

  private Term identity(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("?", args, paracnt, false, 1);
	return paras[0];
  }

  private Term progn(Term args, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	while (args != null) {
	  ret = eval.eval(args);
	  args = args.next;
	}
	return (ret == null ? new ListTerm() : ret);
  }

  private Term cond(Term args, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	while (args != null) {
	  if (args.type != Term.LIST) {
		throw new LScriptException("cond may only contain list arguments");
	  }
	  ListTerm condlist = (ListTerm)args;
	  if (condlist.length() != 2) {
		throw new LScriptException("each cond argument must contain 2 terms");
	  }
	  Term predterm = condlist.getHead();
	  Term execterm = condlist.getTail().getHead();
	  Term result = eval.eval(predterm);
	  if (!result.isNIL()) {
		ret = eval.eval(execterm);
		break;
	  }
	  args = args.next;
	}
	return (ret == null ? new ListTerm() : ret);
  }

  private Term whilefunc(Term args, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	Term predterm = args;
	Term loopbody = args.next;
	try {
	  while (true) {
		//Evaluate predicate
		Term result = eval.eval(predterm);
		if (result.isNIL()) {
		  break;
		}
		//Execute loop body
		Term actarg = loopbody;
		while (actarg != null) {
		  ret = eval.eval(actarg);
		  actarg = actarg.next;
		}
		/* so war's frher
		ListTerm tmp = loopbody;
		while (!tmp.isEmpty()) {
		  ret = eval.eval(tmp.getHead());
		  tmp = tmp.getTail();
		}
		*/
	  }
	} catch (LScriptException e) {
	  if (e.getType() != LScriptException.BREAK) {
		throw e;
	  }
	}
	return (ret == null ? new ListTerm() : ret);
  }

  private Term print(Term args, int paracnt, boolean newline)
  throws LScriptException
  {
	Term ret = null;
	while (args != null) {
	  ret = eval.eval(args);
	  System.out.print(ret.toString());
	  args = args.next;
	}
	if (newline) {
	  System.out.println();
	}
	return (ret == null ? new ListTerm() : ret);
  }

  private Term quitbreak(Term args, int paracnt, int type)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("", args, paracnt, false, 1);
	LScriptException e = new LScriptException("");
	e.setType(type);
	throw e;
  }
}
