/*
 * File........: c:/ARC/PROG/java/gk/lscript/Eval.java
 * Package.....: gk.lscript
 * Created.....: 98/08/08, Guido Krueger
 * RCS.........: $Revision: 1.4 $
 *               $Date: 1998/08/13 23:51:34 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.lscript;

import java.util.*;

/**
 * This class is the LScript term evaluator. To evaluate a term, simply
 * call eval and get the result. Without adding further plugins, the class
 * only understands the setq function and the nil and t symbols. Plugins 
 * might be added using the addPlugin method to add further functions and
 * global variables.
 */
public class Eval
{
  //Instance variables----------------------------------------
  private Vector plugins;
  private Hashtable vars;
  private Hashtable funcs;

  /**
   * Constructs a new Eval object.
   */
  public Eval()
  {
	plugins = new Vector(10);
	vars = new Hashtable(100);
	funcs = new Hashtable(200);
	//create predefined variables
	vars.put("NIL", new ListTerm());
	vars.put("T", new SymbolTerm("t"));
  }

  /**
   * Adds the given plugin to the evaluator. Then, the init method is called
   * and the plugin should register its functions using the registerFunction()
   * method and its global variables using the addVariable() method. When 
   * the evaluator is destroyed, each plugin's destroy() method is called.
   */
  public void addPlugin(LScriptPlugin plugin)
  throws LScriptException
  {
	plugin.init(this);
	plugins.addElement(plugin);
  }

  /**
   * This method should be called by each plugin several times to register
   * all its functions. For each call, a function with the given name is
   * mapped to the given plugin and function number.
   */
  public void registerFunction(
	String name, LScriptPlugin plugin, int funcnum
  )
  {
	funcs.put(name, new FunctionWrapper(plugin, funcnum));
  }

  /**
   * This method should be called by a plugin if it wishes to add a global
   * variable to the var table. 
   */
  public void addVariable(String name, Term value)
  {
	vars.put(name, value);
  }

  /**
   * Evaluates the given term. The result is either non-null, or otherwise
   * a LScriptException is thrown. The result may be a referenced value and
   * thus should be cloned before inserting it into another list.
   */
  public Term eval(Term term)
  throws LScriptException
  {
	String name;
	Term ret = null;
	switch (term.type) {
	case Term.LIST:
	  ListTerm list = (ListTerm) term;
	  if (list.isEmpty()) {
		throw new LScriptException("cannot evaluate empty list");
	  }
	  Term head = list.getHead();
	  if (head.type != Term.SYMBOL) {
		throw new LScriptException(
          "first element of evaluated list must be symbol"
		);
	  }
	  name = head.toString();
	  if (name.equals("setq")) { //------------------------------------
		//setq is a built-in function
		//ret = setq(list.getTail());
		Term varname = head.next;
		if (varname == null || varname.type != Term.SYMBOL) {
		  throw new LScriptException("symbol expected at arg 1");
		}
		Term varvalue = varname.next;
		if (varvalue == null) {
		  throw new LScriptException("value arg missing for setq");
		}
		if (varvalue.next != null) {
		  throw new LScriptException("too many args for setq");
		}
		ret = eval(varvalue);
		vars.put(varname.toString(), ret);
	  } else if (name.equals("quote")) { //------------------------------
		//quote is a built-in function
		if (list.length() != 2) {
		  throw new LScriptException("wrong number of arguments for quote");
		}
		//ret = list.getTail().getHead();
		ret = head.next;
	  } else { //--------------------------------------------------------
		FunctionWrapper w = (FunctionWrapper)funcs.get(name);
		if (w == null) {
		  throw new LScriptException("undefined function: " + name);
		}
		if (w.plugin == null) {
		  //call user-defined function
		  throw new LScriptException(
            "support for user-defined functions not yet implemented"
		  );
		}
		//call plugged-in function
		//ret = w.plugin.execute(w.funcnum, list.getTail());
		ret = w.plugin.execute(w.funcnum, head.next, list.length() - 1);
	  }
	  break;
	case Term.SYMBOL:
	  //lookup name in symbol table
	  name = term.toString();
	  ret = (Term)vars.get(name);
	  if (ret == null) {
		throw new LScriptException("undefined variable: " + name);
	  }
	  break;
	case Term.NUMBER:
	  //A number evaluates to itself
	  ret = term;
	  break;
	case Term.STRING:
	  //A string evaluates to itself
	  ret = term;
	  break;
	default:
	  //Unknown term
	  throw new LScriptException("term not supported");
	}
	return ret;
  }

  /**
   * Calls the destroy() method for all registered plugins.
   */
  public void destroy()
  {
	Enumeration e = plugins.elements();
	while (e.hasMoreElements()) {
	  LScriptPlugin plugin = (LScriptPlugin)e.nextElement();
	  plugin.destroy();
	}
  }

  /**
   */
  public Term[] chkParas(
    String mask, Term paras, int paracnt, boolean openend, int argoffs
  )
  throws LScriptException
  {
	Term ret[];
	int masklen = mask.length();
	if (openend && masklen < 1) {
	  throw new LScriptException("internal error Eval.chkParas.1");
	}
	if (paracnt < masklen) {
	  throw new LScriptException("too few arguments");
	}
	if (paracnt > masklen && !openend) {
	  throw new LScriptException("too many arguments");
	}
	char c, lastc = ' ';
	int offs;
	Term term;
	ret = new Term[paracnt];
	for (int i = 0; i < paracnt; ++i) {
	  term = eval(paras);
	  c = (i < masklen) ? mask.charAt(i) : lastc;
	  lastc = c;
	  offs = i + argoffs;
	  switch (c) {
	  case '?':
		//accept any type
		break;
	  case 'N':
		if (term.type != Term.NUMBER) {
		  throw new LScriptException("number expected at arg " + offs);
		}
		break;
	  case 'S':
		if (term.type != Term.STRING) {
		  throw new LScriptException("string expected at arg " + offs);
		}
		break;
	  case 'Y':
		if (term.type != Term.SYMBOL) {
		  throw new LScriptException("symbol expected at arg " + offs);
		}
		break;
	  case 'L':
		if (term.type != Term.LIST) {
		  throw new LScriptException("list expected at arg " + offs);
		}
		break;
	  default:
		throw new LScriptException("internal error Eval.chkParas.2");
	  }
	  ret[i] = term;
	  paras = paras.next;
	}
	return ret;
  }
}

/**
 * Wrapper for function definitions. Holds both plugged-in functions (plugin
 * is non-null) and user-defined functions (plugin is null).
 */
class FunctionWrapper
{
  //Instance variables
  public LScriptPlugin plugin;
  public int           funcnum;

  public FunctionWrapper(LScriptPlugin plugin, int funcnum)
  {
	this.plugin = plugin;
	this.funcnum = funcnum;
  }
}
