/*
 * File........: c:/ARC/PROG/java/gk/lscript/NumberPlugin.java
 * Package.....: gk.lscript
 * Created.....: 98/08/09, Guido Krueger
 * RCS.........: $Revision: 1.4 $
 *               $Date: 1998/08/13 23:51:34 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.lscript;

public class NumberPlugin
extends LScriptPlugin
{
  //Function numbers constants----------------------------------------
  private final int FUNC_PLUS      =  1;
  private final int FUNC_MINUS     =  2;
  private final int FUNC_TIMES     =  3;
  private final int FUNC_DIVIDE    =  4;
  private final int FUNC_REMAINDER =  5;
  private final int FUNC_1PLUS     =  6;
  private final int FUNC_1MINUS    =  7;
  private final int FUNC_ABS       =  8;
  private final int FUNC_SQRT      =  9;
  private final int FUNC_INT       = 10;
  private final int FUNC_FRAC      = 11;
  private final int FUNC_POW       = 12;
  private final int FUNC_INV       = 13;
  private final int FUNC_ROUND     = 14;
  private final int FUNC_SIN       = 15;
  private final int FUNC_COS       = 16;
  private final int FUNC_TAN       = 17;
  private final int FUNC_ASIN      = 18;
  private final int FUNC_ACOS      = 19;
  private final int FUNC_ATAN      = 20;
  private final int FUNC_RANDOM    = 21;
  private final int FUNC_LOG       = 22;
  private final int FUNC_EXP       = 23;
  private final int FUNC_MAX       = 24;
  private final int FUNC_MIN       = 25;

  //Instance variables----------------------------------------
  protected Eval eval;

  public NumberPlugin()
  {
  }

  public void init(Eval eval)
  throws LScriptException
  {
	this.eval = eval;
	//Functions
	eval.registerFunction("+",        this, FUNC_PLUS);
	eval.registerFunction("-",        this, FUNC_MINUS);
	eval.registerFunction("*",        this, FUNC_TIMES);
	eval.registerFunction("/",        this, FUNC_DIVIDE);
	eval.registerFunction("%",        this, FUNC_REMAINDER);
	eval.registerFunction("add1",     this, FUNC_1PLUS);
	eval.registerFunction("sub1",     this, FUNC_1MINUS);
	eval.registerFunction("abs",      this, FUNC_ABS);
	eval.registerFunction("sqrt",     this, FUNC_SQRT);
	eval.registerFunction("int",      this, FUNC_INT);
	eval.registerFunction("frac",     this, FUNC_FRAC);
	eval.registerFunction("pow",      this, FUNC_POW);
	eval.registerFunction("inv",      this, FUNC_INV);
	eval.registerFunction("round",    this, FUNC_ROUND);
	eval.registerFunction("sin",      this, FUNC_SIN);
	eval.registerFunction("cos",      this, FUNC_COS);
	eval.registerFunction("tan",      this, FUNC_TAN);
	eval.registerFunction("asin",     this, FUNC_ASIN);
	eval.registerFunction("acos",     this, FUNC_ACOS);
	eval.registerFunction("atan",     this, FUNC_ATAN);
	eval.registerFunction("random",   this, FUNC_RANDOM);
	eval.registerFunction("log",      this, FUNC_LOG);
	eval.registerFunction("exp",      this, FUNC_EXP);
	eval.registerFunction("max",      this, FUNC_MAX);
	eval.registerFunction("min",      this, FUNC_MIN);
	//Variables
	eval.addVariable("PI", new NumberTerm(Math.PI));
	eval.addVariable("E", new NumberTerm(Math.E));
  }

  public Term execute(int funcnum, Term paras, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	switch (funcnum) {
	case FUNC_PLUS:
	  ret = seriesmath(paras, paracnt, '+');
	  break;
	case FUNC_MINUS:
	  ret = seriesmath(paras, paracnt, '-');
	  break;
	case FUNC_TIMES:
	  ret = seriesmath(paras, paracnt, '*');
	  break;
	case FUNC_DIVIDE:
	  ret = seriesmath(paras, paracnt, '/');
	  break;
	case FUNC_REMAINDER:
	  ret = seriesmath(paras, paracnt, '%');
	  break;
	case FUNC_1PLUS:
	  ret = inc(paras, paracnt, 1);
	  break;
 	case FUNC_1MINUS:
	  ret = inc(paras, paracnt, -1);
	  break;
 	case FUNC_ABS:
	  ret = abs(paras, paracnt);
	  break;
 	case FUNC_SQRT:
	  ret = sqrt(paras, paracnt);
	  break;
 	case FUNC_INT:
	  ret = toint(paras, paracnt);
	  break;
 	case FUNC_FRAC:
	  ret = frac(paras, paracnt);
	  break;
 	case FUNC_POW:
	  ret = pow(paras, paracnt);
	  break;
 	case FUNC_INV:
	  ret = inv(paras, paracnt);
	  break;
 	case FUNC_ROUND:
	  ret = round(paras, paracnt);
	  break;
 	case FUNC_SIN:  case FUNC_COS:  case FUNC_TAN:
 	case FUNC_ASIN: case FUNC_ACOS: case FUNC_ATAN:
	  ret = trig(paras, paracnt, funcnum);
	  break;
 	case FUNC_RANDOM:
	  ret = random(paras, paracnt);
	  break;
 	case FUNC_LOG:
	  ret = log(paras, paracnt);
	  break;
 	case FUNC_EXP:
	  ret = exp(paras, paracnt);
	  break;
	case FUNC_MAX:
	  ret = seriesmath(paras, paracnt, 'A');
	  break;
	case FUNC_MIN:
	  ret = seriesmath(paras, paracnt, 'I');
	  break;
	default:
	  throw new LScriptException(
        "internal error: unknown function number in NumberPlugin"
	  );
	}
	return ret;
  }

  public void destroy()
  {
  }

  //Function implementations----------------------------------------
  private Term seriesmath(Term args, int paracnt, char op)
  throws LScriptException
  {
	int intresult = 0;
	double doubleresult = 0.0;
	boolean isdouble = false;
	Term[] paras = eval.chkParas("N", args, paracnt, true, 1);
	for (int i = 0; i < paras.length; ++i) {
	  NumberTerm num = (NumberTerm) paras[i];
	  if (!isdouble && !num.isInt()) {
		isdouble = true;
		doubleresult = intresult;
	  }
	  if (isdouble) {
		if (i == 0) {
		  doubleresult = num.getAsDouble();
		} else if (op == '+') {
		  doubleresult += num.getAsDouble();
		} else if (op == '-') {
		  doubleresult -= num.getAsDouble();
		} else if (op == '*') {
		  doubleresult *= num.getAsDouble();
		} else if (op == '/') {
		  doubleresult /= num.getAsDouble();
		} else if (op == '%') {
		  doubleresult %= num.getAsDouble();
		} else if (op == 'A') {
		  double val = num.getAsDouble();
		  if (val > doubleresult) {
			doubleresult = val;
		  }
		} else if (op == 'I') {
		  double val = num.getAsDouble();
		  if (val < doubleresult) {
			doubleresult = val;
		  }
		}
	  } else {
		if (i == 0) {
		  intresult = num.getAsInt();
		} else if (op == '+') {
		  intresult += num.getAsInt();
		} else if (op == '-') {
		  intresult -= num.getAsInt();
		} else if (op == '*') {
		  intresult *= num.getAsInt();
		} else if (op == '/') {
		  intresult /= num.getAsInt();
		} else if (op == '%') {
		  intresult %= num.getAsInt();
		} else if (op == 'A') {
		  int val = num.getAsInt();
		  if (val > intresult) {
			intresult = val;
		  }
		} else if (op == 'I') {
		  int val = num.getAsInt();
		  if (val < intresult) {
			intresult = val;
		  }
		}
	  }
	}
	if (isdouble) {
	  return new NumberTerm(doubleresult);
	} else {
	  return new NumberTerm(intresult);
	}
  }

  private Term inc(Term args, int paracnt, int value)
  throws LScriptException
  {
	/*
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	*/
	//Handoptimierung mit jprobe hat die Laufzeit um 50% verringert
	if (args == null) {
	  throw new LScriptException("too few arguments");
	}
	if (args.next != null) {
	  throw new LScriptException("too many arguments");
	}
	Term tmp = eval.eval(args);
	if (tmp.type != Term.NUMBER) {
	  throw new LScriptException("number expected at arg 1");
	}
	NumberTerm num = (NumberTerm)tmp;
	if (num.isInt()) {
	  return new NumberTerm(num.getAsInt() + value);
	} else {
	  return new NumberTerm(num.getAsDouble() + value);
	}
  }

  private Term abs(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	if (num.isInt()) {
	  return new NumberTerm(Math.abs(num.getAsInt()));
	} else {
	  return new NumberTerm(Math.abs(num.getAsDouble()));
	}
  }

  private Term sqrt(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm(Math.sqrt(num.getAsDouble()));
  }

  private Term toint(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm(num.getAsInt());
  }

  private Term frac(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	double value = num.getAsDouble();
	if (value >= 0) {
	  value = value - Math.floor(value);
	} else {
	  value = value - Math.ceil(value);
	}
	return new NumberTerm(Math.abs(value));
  }

  private Term pow(Term args, int paracnt)
  throws LScriptException
  {
	NumberTerm ret = null;
	Term[] paras = eval.chkParas("NN", args, paracnt, false, 1);
	NumberTerm num1 = (NumberTerm)paras[0];
	NumberTerm num2 = (NumberTerm)paras[1];
	if (num1.isInt() && num2.isInt()) {
	  //compute whole power
	  int result = 1;
	  int x = num1.getAsInt();
	  int y = num2.getAsInt();
	  for (int i = 0; i < y; ++i) {
		if (result >= Integer.MAX_VALUE / x) {
		  throw new LScriptException("numeric overflow");
		}
		result *= x;
	  }
	  ret = new NumberTerm(result);
	} else {
	  double x = num1.getAsDouble();
	  double y = num2.getAsDouble();
	  ret = new NumberTerm(Math.exp(y * Math.log(x)));
	}
	return ret;
  }

  private Term inv(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm(1.0 / num.getAsDouble());
  }

  private Term round(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm((int)Math.round(num.getAsDouble()));
  }

  private Term trig(Term args, int paracnt, int funcnum)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	double value = num.getAsDouble();
	if (funcnum == FUNC_SIN) {
	  value = Math.sin(value);
	} else if (funcnum == FUNC_COS) {
	  value = Math.cos(value);
	} else if (funcnum == FUNC_TAN) {
	  value = Math.tan(value);
	} else if (funcnum == FUNC_ASIN) {
	  value = Math.asin(value);
	} else if (funcnum == FUNC_ACOS) {
	  value = Math.acos(value);
	} else if (funcnum == FUNC_ATAN) {
	  value = Math.atan(value);
	}
	return new NumberTerm(value);
  }

  private Term random(Term args, int paracnt)
  throws LScriptException
  {
	NumberTerm ret = null;
	if (paracnt == 0) {
	  //create floating point number betwenn 0.0 and 1.0
	  ret = new NumberTerm(Math.random());
	} else {
	  //create integer with the given maximum
	  Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	  NumberTerm num = (NumberTerm)paras[0];
	  ret = new NumberTerm((int)(Math.random() * num.getAsInt()));
	}
	return ret;
  }

  private Term log(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm(Math.log(num.getAsDouble()));
  }

  private Term exp(Term args, int paracnt)
  throws LScriptException
  {
	Term[] paras = eval.chkParas("N", args, paracnt, false, 1);
	NumberTerm num = (NumberTerm)paras[0];
	return new NumberTerm(Math.exp(num.getAsDouble()));
  }
}
