/*
 * File........: c:/ARC/PROG/java/gk/lscript/Parser.java
 * Package.....: gk.lscript
 * Created.....: 98/08/08, Guido Krueger
 * RCS.........: $Revision: 1.3 $
 *               $Date: 1998/08/13 22:26:19 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.lscript;

import java.io.*;

/**
 * Parser for syntactical analysis of LScript language.
 */
public class Parser
{
  //Instance variables----------------------------------------
  protected Scanner scanner;
  protected int     actsym;

  /**
   * Creates a new LScript parser. The input argument provides a Reader
   * with the LScript code.
   */
  public Parser(Reader input)
  {
	scanner = new Scanner(input);
	actsym = scanner.nextSym();
  }

  /**
   * Returns the next term from the input reader. If the end of the input
   * Reader is reached, the method returns null. If an error occurs, a
   * LScriptException is thrown. The return value is always a new Term
   * object which is not yet aliased elsewhere.
   */
  public Term nextTerm()
  throws LScriptException
  {
	Term tmp, ret = null;

	//ignore comments
	while (actsym == Scanner.COMMENT) {
	  actsym = scanner.nextSym();
	}
	if (actsym == Scanner.INTEGER) {
	  ret = new NumberTerm(scanner.getIntData());
	  actsym = scanner.nextSym();
	} else if (actsym == Scanner.DOUBLE) {
	  ret = new NumberTerm(scanner.getDoubleData());
	  actsym = scanner.nextSym();
	} else if (actsym == Scanner.STRING) {
	  ret = new StringTerm(scanner.getStringData());
	  actsym = scanner.nextSym();
	} else if (actsym == Scanner.IDENTIFIER) {
	  ret = new SymbolTerm(scanner.getStringData());
	  actsym = scanner.nextSym();
	} else if (actsym == Scanner.QUOTE) {
	  ListTerm list = new ListTerm();
	  list.addElement(new SymbolTerm("quote"));
	  actsym = scanner.nextSym();
	  if ((tmp = nextTerm()) == null) {
		throwError("unexpected end of input");
	  }
	  list.addElement(tmp);
	  ret = list;
	} else if (actsym == Scanner.LPAREN) {
	  ListTerm list = new ListTerm();
	  actsym = scanner.nextSym();
	  while (true) {
		if (actsym == Scanner.RPAREN) {
		  actsym = scanner.nextSym();
		  break;
		} else {
		  if ((tmp = nextTerm()) == null) {
			throwError("unexpected end of input");
		  }
		  list.addElement(tmp);
		}
	  }
	  ret = list;
	} else if (actsym == Scanner.RPAREN) {
	  throwError("unexpected symbol: )");
	} else if (actsym == Scanner.ERROR) {
	  throwError(scanner.getError());
	} else if (actsym == Scanner.EOF) {
	  ret = null;
	} else {
	  throwError("unknown symbol: " + scanner.getSymbolName(actsym));
	}
	return ret;
  }

  /**
   * Returns the current input line.
   */
  public int getLine()
  {
	return scanner.getLine();
  }

  //---Private methods-----------------------------------------------------
  private void throwError(String msg)
  throws LScriptException
  {
	LScriptException e = new LScriptException(msg);
	e.setLine(scanner.getLine());
	throw e;
  }

  //---main method for testing only----------------------------------------
  public static void main(String args[])
  {
	if (args.length != 1) {
	  System.out.println("usage: java Parser (lisp-prog | @lisp-file)");
	  System.exit(1);
	}
	Parser parser = null;
	if (args[0].startsWith("@")) {
	  try {
		parser = new Parser(new FileReader(args[0].substring(1)));
	  } catch (FileNotFoundException e) {
		System.out.println(e.toString());
		System.exit(1);
	  }
	} else {
	  parser = new Parser(new StringReader(args[0]));
	}
	try {
	  while (true) {
		Term term = parser.nextTerm();
		if (term == null) {
		  System.out.println("<eof>");
		  break;
		}
		System.out.println(term.toString());
	  }
	} catch (LScriptException e) {
	  System.out.println(e.toString());
	}
  }
}
