/*
 * File........: c:/ARC/PROG/java/gk/lscript/PredicatePlugin.java
 * Package.....: gk.lscript
 * Created.....: 98/08/09, Guido Krueger
 * RCS.........: $Revision: 1.4 $
 *               $Date: 1998/08/15 01:17:36 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.lscript;

public class PredicatePlugin
extends LScriptPlugin
{
  //Function numbers constants----------------------------------------
  private final int FUNC_LT       = 1;
  private final int FUNC_LTE      = 2;
  private final int FUNC_GT       = 3;
  private final int FUNC_GTE      = 4;
  private final int FUNC_EQ       = 5;
  private final int FUNC_NEQ      = 6;
  private final int FUNC_NOT      = 7;
  private final int FUNC_AND      = 8;
  private final int FUNC_OR       = 9;

  //Instance variables----------------------------------------
  protected Eval eval;

  public PredicatePlugin()
  {
  }

  public void init(Eval eval)
  throws LScriptException
  {
	this.eval = eval;
	eval.registerFunction("<",        this, FUNC_LT);
	eval.registerFunction("<=",       this, FUNC_LTE);
	eval.registerFunction(">",        this, FUNC_GT);
	eval.registerFunction(">=",       this, FUNC_GTE);
	eval.registerFunction("=",        this, FUNC_EQ);
	eval.registerFunction("!=",       this, FUNC_NEQ);
	eval.registerFunction("not",      this, FUNC_NOT);
	eval.registerFunction("and",      this, FUNC_AND);
	eval.registerFunction("or",       this, FUNC_OR);
  }

  public Term execute(int funcnum, Term paras, int paracnt)
  throws LScriptException
  {
	Term ret = null;
	switch (funcnum) {
	case FUNC_LT: case FUNC_LTE:
	case FUNC_GT: case FUNC_GTE:
	case FUNC_EQ: case FUNC_NEQ:
	  ret = compare(paras, paracnt, funcnum);
	  break;
	case FUNC_NOT:
	  ret = not(paras, paracnt);
	  break;
	case FUNC_AND:
	  ret = and(paras, paracnt);
	  break;
	case FUNC_OR:
	  ret = or(paras, paracnt);
	  break;
	default:
	  throw new LScriptException(
        "internal error: unknown function number in PredicatePlugin"
	  );
	}
	return ret;
  }

  public void destroy()
  {
  }

  //Function implementations----------------------------------------
  private Term compare(Term args, int paracnt, int funcnum)
  throws LScriptException
  {
	boolean ret = false;
	if (paracnt != 2) {
	  throw new LScriptException("wrong number of arguments");
	}
	Term t1 = eval.eval(args);
	Term t2 = eval.eval(args.next);
	//System.out.println("t1=" + t1.toString() + " t2=" + t2.toString());
	if (t1.type == Term.NUMBER && t2.type == Term.NUMBER) {
	  NumberTerm num1 = (NumberTerm)t1;
	  NumberTerm num2 = (NumberTerm)t2;
	  if (!num1.isInt() || !num2.isInt()) {
		if (funcnum == FUNC_LT) {
		  ret = num1.getAsDouble() < num2.getAsDouble();
		} else if (funcnum == FUNC_LTE) {
		  ret = num1.getAsDouble() <= num2.getAsDouble();
		} else if (funcnum == FUNC_GT) {
		  ret = num1.getAsDouble() > num2.getAsDouble();
		} else if (funcnum == FUNC_GTE) {
		  ret = num1.getAsDouble() >= num2.getAsDouble();
		} else if (funcnum == FUNC_EQ) {
		  ret = num1.getAsDouble() == num2.getAsDouble();
		} else if (funcnum == FUNC_NEQ) {
		  ret = num1.getAsDouble() != num2.getAsDouble();
		}
	  } else {
		if (funcnum == FUNC_LT) {
		  ret = num1.getAsInt() < num2.getAsInt();
		} else if (funcnum == FUNC_LTE) {
		  ret = num1.getAsInt() <= num2.getAsInt();
		} else if (funcnum == FUNC_GT) {
		  ret = num1.getAsInt() > num2.getAsInt();
		} else if (funcnum == FUNC_GTE) {
		  ret = num1.getAsInt() >= num2.getAsInt();
		} else if (funcnum == FUNC_EQ) {
		  ret = num1.getAsInt() == num2.getAsInt();
		} else if (funcnum == FUNC_NEQ) {
		  ret = num1.getAsInt() != num2.getAsDouble();
		}
	  }
	} else if (t1.type == Term.STRING && t2.type == Term.STRING) {
	  if (funcnum == FUNC_LT) {
		ret = t1.toString().compareTo(t2.toString()) < 0;
	  } else if (funcnum == FUNC_LTE) {
		ret = t1.toString().compareTo(t2.toString()) <= 0;
	  } else if (funcnum == FUNC_GT) {
		ret = t1.toString().compareTo(t2.toString()) > 0;
	  } else if (funcnum == FUNC_GTE) {
		ret = t1.toString().compareTo(t2.toString()) >= 0;
	  } else if (funcnum == FUNC_EQ) {
		ret = t1.toString().equals(t2.toString());
	  } else if (funcnum == FUNC_NEQ) {
		ret = !t1.toString().equals(t2.toString());
	  }
	} else {
	  throw new LScriptException(
        "arguments must be either both NUMBERS or STRINGS"
	  );
	}
	if (ret) {
	  return new SymbolTerm("T");
	} else {
	  return new ListTerm();
	}
  }

  private Term not(Term args, int paracnt)
  throws LScriptException
  {
	if (paracnt != 1) {
	  throw new LScriptException("wrong number of arguments");
	}
	Term t1 = eval.eval(args);
	if (t1.isNIL()) {
	  return new SymbolTerm("T");
	} else {
	  return new ListTerm();
	}
  }

  private Term and(Term args, int paracnt)
  throws LScriptException
  {
	if (paracnt < 1) {
	  throw new LScriptException("too few arguments");
	}
	boolean ret = true;
	while (args != null) {
	  Term t1 = eval.eval(args);
	  if (t1.isNIL()) {
		ret = false;
		break;
	  }
	  args = args.next;
	}
	if (ret) {
	  return new SymbolTerm("T");
	} else {
	  return new ListTerm();
	}
  }

  private Term or(Term args, int paracnt)
  throws LScriptException
  {
	if (paracnt < 1) {
	  throw new LScriptException("too few arguments");
	}
	boolean ret = false;
	while (args != null) {
	  Term t1 = eval.eval(args);
	  if (!t1.isNIL()) {
		ret = true;
		break;
	  }
	  args = args.next;
	}
	if (ret) {
	  return new SymbolTerm("T");
	} else {
	  return new ListTerm();
	}
  }
}
