/*
 * File........: c:/ARC/PROG/java/gk/util/MatrixWriter.java
 * Package.....: gk.util
 * Created.....: 98/06/02, Guido Krueger
 * RCS.........: $Revision: 1.5 $
 *               $Date: 1998/07/08 14:09:54 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.util;

import java.io.*;

/**
 * The MatrixWriter class implements a kind of a crossbar switch. Each
 * row represents a Writer and each column represents a channel. By
 * putting switches either on or off at the intersections of rows and
 * columns, an arbitrary combination of channels and writers could be 
 * connected. For each output method, a particular channel is specified
 * and output is then multicasted to all Writers which are currently 
 * connected to the channel. For each writer method, there is also a
 * "channel-less" version, which use the default channel 0.
 */
public class MatrixWriter
extends Writer
{
  protected Writer writers[];
  protected boolean crossbar[][];
  protected int numchannels;
  protected int numwriters;

  /**
   * Creates a new MatrixWriter with the given number of channels
   * and writers.
   */
  public MatrixWriter(int numchannels, int numwriters)
  {
	this.numwriters = numwriters;
	this.numchannels = numchannels;
	writers = new Writer[numwriters];
	crossbar = new boolean[numwriters][numchannels];
	for (int i = 0; i < numwriters; ++i) {
	  writers[i] = null;
	  for (int j = 0; j < numchannels; ++j) {
		crossbar[i][j] = false;
	  }
	}
  }

  /**
   * Installs the given writer at the specified crossbar row. All
   * crossbar switches remain unchanged.
   */
  public void setWriter(int writerpos, Writer writer)
  {
	writers[writerpos] = writer;
  }

  /**
   * Returns the writer at the given crossbar row.
   */
  public Writer getWriter(int writerpos)
  {
	return writers[writerpos];
  }

  /**
   * Connects the crossbar switch for the given channel and writer.
   */
  public void connect(int channel, int writer)
  {
	crossbar[writer][channel] = true;
  }

  /**
   * Disconnects the crossbar switch for the given channel and writer.
   */
  public void disconnect(int channel, int writer)
  {
	crossbar[writer][channel] = false;
  }

  /**
   * Returns the state of the crossbar switch for the given channel
   * and writer.
   */
  public boolean isConnected(int channel, int writer)
  {
	return crossbar[writer][channel];
  }

  /**
   * Write a single character to the default channel 0.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(int c) 
  throws IOException 
  {
	this.write(0, c);
  }

  /**
   * Write a single character to the given channel.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(int channel, int c)
  throws IOException 
  {
	for (int i = 0; i < numwriters; ++i) {
	  if (crossbar[i][channel]) {
		writers[i].write(c);
	  }
	}
  }

  /**
   * Write a portion of an array of characters to the default 
   * channel 0.
   *
   * @param  cbuf  Buffer of characters to be written
   * @param  off   Offset from which to start reading characters
   * @param  len   Number of characters to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(char cbuf[], int off, int len)
  throws IOException 
  {
	this.write(0, cbuf, off, len);
  }
  
  /**
   * Write a portion of an array of characters to the given
   * channel.
   *
   * @param  cbuf  Buffer of characters to be written
   * @param  off   Offset from which to start reading characters
   * @param  len   Number of characters to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(int channel, char cbuf[], int off, int len)
  throws IOException 
  {
	for (int i = 0; i < numwriters; ++i) {
	  if (crossbar[i][channel]) {
		writers[i].write(cbuf, off, len);
	  }
	}
  }
  
  /**
   * Write a portion of a string to the default channel 0.
   *
   * @param  str  String to be written
   * @param  off  Offset from which to start reading characters
   * @param  len  Number of characters to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(String str, int off, int len)
  throws IOException 
  {
	this.write(0, str, off, len);
  }

  /**
   * Write a portion of a string to the given channel.
   *
   * @param  str  String to be written
   * @param  off  Offset from which to start reading characters
   * @param  len  Number of characters to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(int channel, String str, int off, int len)
  throws IOException 
  {
	for (int i = 0; i < numwriters; ++i) {
	  if (crossbar[i][channel]) {
		writers[i].write(str, off, len);
	  }
	}
  }

  /**
   * Write a string to the default channel 0.
   *
   * @param  str  String to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(String str)
  throws IOException 
  {
	this.write(0, str, 0, str.length());
  }

  /**
   * Write a string to the given channel.
   *
   * @param  str  String to be written
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void write(int channel, String str)
  throws IOException 
  {
	this.write(channel, str, 0, str.length());
  }

  /**
   * Flushes all writers.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void flush()
  throws IOException 
  {
	for (int i = 0; i < numwriters; ++i) {
	  writers[i].flush();
	}
  }

  /**
   * Flushes a single writer.
   *
   * @exception  IOException  If an I/O error occurs
   */
  public void flushWriter(int writer)
  throws IOException 
  {
	writers[writer].flush();
  }

  /**
   * Closes all writers.
   */
  public void close()
  {
	for (int i = 0; i < numwriters; ++i) {
	  try {
		writers[i].close();
	  } catch (IOException e) {
		//ignore: a closing exception is considered harmless here
	  }
	}
  }

  //---Test code starts here----------------------------------------
  public static void main(String args[])
  throws Exception
  {
	MatrixWriter mw = new MatrixWriter(4, 3);
	mw.setWriter(0, new OutputStreamWriter(System.out));
	mw.setWriter(1, new FileWriter("x1.tmp"));
	mw.setWriter(2, new FileWriter("x2.tmp"));
	mw.connect(0, 0); mw.connect(0, 2);
	mw.connect(1, 1); mw.connect(1, 2);
	mw.connect(3, 0); mw.connect(3, 1); mw.connect(3, 2);
	mw.write(0, "Channel 0\r\n", 0, 11);
	mw.write(1, "Channel 1\r\n", 0, 11);
	mw.write(2, "Channel 2\r\n", 0, 11);
	mw.write(3, "Channel 3\r\n", 0, 11);
	mw.write("This is also channel 0\r\n", 0, 24);
	mw.close();
  }
}
