/*
 * File........: c:/ARC/PROG/java/gk/util/Queue.java
 * Package.....: gk.util
 * Created.....: 98/07/28, Guido Krueger
 * RCS.........: $Revision: 1.3 $
 *               $Date: 1998/07/29 01:26:27 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.util;

import java.util.*;

/**
 * This class implements the general data structure of a queue (which is
 * missing in both JDK 1.1 data structures as well as 1.2 collections).
 * A queue is a dynamic list of elements with a restricted set of operations:
 * <ul>
 * <li>A new element is inserted at the back of the queue.
 * <li>An existing element may only be removed from the front of the queue.
 * </ul>
 * A queue thus implements the well-known first-in-first-out access scheme.
 * This implementation is based on a linear list (of ElementWrapper objects)
 * which neither requires costly array copy operations nor has a fixed 
 * size limit:
 * <pre>
 *          last                         first
 *           |                             |
 *           |                             |
 *           v                             v
 *        +-----+   +-----+   +-----+   +-----+
 *     |--|     |<--|     |<--|     |<--|     |
 *        +-----+   +-----+   +-----+   +-----+
 * </pre>
 * 
 * All basic methods are O(1), undoInsertions is usually O(size()).
 */
public class Queue
{
  protected ElementWrapper first;
  protected ElementWrapper last;
  protected int count;

  /**
   * Creates an empty queue.
   */
  public Queue()
  {
	first = last = null;
	count = 0;
  }

  /**
   * Adds the element o to the end of the queue. Returns true if successful,
   * otherwise false is returned.
   */
  public boolean add(Object o)
  {
	if (count == 0) {
	  //insert first element
	  first = new ElementWrapper();
	  last = first;
	  count = 1;
	} else {
	  //insert element into non-empty queue
	  last.next = new ElementWrapper();
	  last = last.next;
	  ++count;
	}
	last.element = o;
	last.next = null;
	return true;
  }

  /**
   * Removes the first element from the queue and returns it. If the 
   * queue is empty, null is returned and the queue remains unchanged.
   */
  public Object remove()
  {
	ElementWrapper ret = first;
	if (count > 0) { //queue non-empty
	  --count;
	  first = first.next;
	  if (first == null) {
		last = null;
		count = 0;
	  }
	}
	return ret.element;
  }

  /**
   * Returns the number of elements in the queue.
   */
  public int size()
  {
	return count;
  }

  /**
   * Returns true if the queue is empty. Otherwise, false is returned.
   */
  public boolean isEmpty()
  {
	return count == 0;
  }

  /**
   * Removes all elements from the queue.
   */
  public void clear()
  {
	while (first != null) {
	  ElementWrapper tmp = first;
	  first = first.next;
	  tmp = null;
	}
	first = last = null;
	count = 0;
  }

  /**
   * Undoes the last undocnt number of insertions (e.g. removes this
   * number of elements from the end of the queue). If there are fewer
   * than undocnt elements in the queue, clear is called.
   */
  public void undoInsertions(int undocnt)
  {
	if (undocnt >= count) {
	  //remove all elements
	  clear();
	} else {
	  //Advance to new last element
	  ElementWrapper tmp = first;
	  for (int i = undocnt; i < count - 1; ++i) {
		tmp = tmp.next;
	  }
	  //Unchain tail
	  last = tmp;
	  last.next = null;
	  count -= undocnt;
	}
  }

  /**
   * Returns an enumeration of all the elements in the queue with the
   * queue ordering preserved. Thus, the front element is returned first
   * and the last element is returned last.
   */
  public Enumeration elements()
  {
	return new Enumeration() {
	  ElementWrapper tmp = first;

	  public boolean hasMoreElements()
	  {
		return tmp != null;
	  }

	  public Object nextElement() 
	  {
		if (tmp != null) {
		  Object ret = tmp.element;
		  tmp = tmp.next;
		  return ret;
		}
		throw new NoSuchElementException("Queue Enumeration");
	  }
	};
  }

  //--- class ElementWrapper ------------------------------
  /**
   * Wrapper for queue elements.
   */
  class ElementWrapper
  {
	public Object element;
	public ElementWrapper next;
  }

  //------------------------------------------------------------
  /**
   * Testing purposes only.
   */
  public static void main(String args[])
  {
	Queue q1 = new Queue();
	Queue q2 = new Queue();
	q1.add(new Integer(1));
	q1.add(new Integer(2));
	q1.add(new Integer(3));
	q1.add(new Integer(4));
	q1.remove();
	q1.add(new Integer(5));
	System.out.println("q1:");
	while (!q1.isEmpty()) {
	  Integer i = (Integer)q1.remove();
	  System.out.println(i.toString());
	  q2.add(i);
	}
	q2.undoInsertions(2);
	q2.add(new Integer(6));
	q2.add(new Integer(7));
	System.out.println("q2:");
	Enumeration e = q2.elements();
	while (e.hasMoreElements()) {
	  Integer i = (Integer)e.nextElement();
	  System.out.println(i.toString());
	}
  }
}
