/*
 * File........: c:/ARC/PROG/java/gk/util/SortableVector.java
 * Package.....: gk.util
 * Created.....: 98/07/11, Guido Krueger
 * RCS.........: $Revision: 1.6 $
 *               $Date: 1998/08/27 09:42:22 $ $Author: guido $
 *
 * Copyright (c) 1998 Guido Krueger. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies.
 *
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE FOR ANY 
 * DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING 
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package gk.util;

import java.util.*;

/**
 * An extension to the java.util.Vector class which provides methods 
 * to sort the elements using either selection sort or quicksort. The
 * selection sort is pretty simple and quite slow (O(N^2)). It is
 * virtually unusable for more than several hundred elements. The 
 * quicksort, on the other hand, performs quite well. In a number of 
 * simulations it has been able to sort 100.000 double values in a 
 * single second (using JDK1.2Beta4 with JIT enabled). The runtime has 
 * always approximated N log N, no degenerated cases appeared during 
 * testing.
 */
public class SortableVector
extends Vector
{
  /**
   * Calls the corresponding base class constructor.
   */
  public SortableVector()
  {
	super();
  }

  /**
   * Calls the corresponding base class constructor.
   */
  public SortableVector(int initialCapacity)
  {
	super(initialCapacity);
  }

  /**
   * Calls the corresponding base class constructor.
   */
  public SortableVector(int initialCapacity, int capacityIncrement)
  {
	super(initialCapacity, capacityIncrement);
  }

  /**
   * Sorts the elements of the vector using a simple O(n^2) selection
   * sort. The comp argument must be an object which implements the 
   * ElementComparator interface. See test code below for an example of
   * how to use it.
   */
  public void sort(ElementComparator comp)
  {
	int smallest;
	for (int i = 0; i < elementCount - 1; ++i) {
	  //find smallest remaining element
	  smallest = i;
	  for (int j = i + 1; j < elementCount; ++j) {
		if (comp.preceeds(elementData[j], elementData[smallest])) {
		  smallest = j;
		}
	  }
	  //exchange smallest and i
	  if (smallest != i) {
		Object tmp = elementData[i];
		elementData[i] = elementData[smallest];
		elementData[smallest]= tmp;
	  }
	}
  }

  /**
   * Sorts the elements of the vector using a recursive N log N quicksort. 
   * The pivot is selected randomly, so even in the case of an already
   * sorted vector, runtime is usually comparable to the average case.
   * The comp argument must be an object which implements the 
   * ElementComparator interface. See test code below for an example of
   * how to use it.
   */
  public void qsort(ElementComparator comp)
  {
	qsortInternal(0, elementCount - 1, comp);
  }

  //---Internal methods---------------------------------------------------
  /**
   * Internally used recursive quicksort procedure.
   */
  private void qsortInternal(int left, int right, ElementComparator comp)
  {
	Object o1, o2, pivot;
	if (left < right) {
	  if (right - left == 1) { //Directly sort 2 elements
		o1 = elementData[left];
		o2 = elementData[right];
		if (!comp.preceeds(o1, o2)) {
		  //swap elements
		  elementData[left] = o2;
		  elementData[right] = o1;
		}
	  } else {
		//Randomly choose pivot element
		int pos = left + (int)((right - left + 1) * Math.random());
		pivot = elementData[pos];
		int i = left - 1;
		int j = right + 1;
		//Start partitioning data
		while (true) {
		  do { //Look for left border
			o1 = elementData[++i];
		  } while (comp.preceeds(o1, pivot));
		  do { //Look for right border
			o2 = elementData[--j];
		  } while (comp.preceeds(pivot, o2));
		  if (i >= j) {
			//Partioning done, exit loop
			break;
		  }
		  //Swap border elements
		  elementData[i] = o2;
		  elementData[j] = o1;
		}
		//Recursively qsort partition[s]
		if (j == right) {
		  qsortInternal(left, j - 1, comp);
		} else {
		  qsortInternal(left, j, comp);
		  qsortInternal(j + 1, right, comp);
		}
	  }
	}
  }

  //------------------------------------------------------------
  /**
   * The following code is for testing purposes only.
   */
  public static void main(String args[])
  {
	extensiveTest();
	System.exit(0);
	//Create new vector
	SortableVector v = new SortableVector();
	System.out.println("Creating vector...");
	v.addElement("completely unsorted: 3");
	v.addElement("completely unsorted: 5");
	v.addElement("completely unsorted: 2");
	v.addElement("completely unsorted: 1");
	v.addElement("completely unsorted: 4");
	v.addElement("already sorted: 1");
	v.addElement("already sorted: 2");
	v.addElement("already sorted: 3");
	v.addElement("already sorted: 4");
	v.addElement("already sorted: 5");
    v.addElement("descending: 5");
	v.addElement("descending: 4");
	v.addElement("descending: 3");
	v.addElement("descending: 2");
	v.addElement("descending: 1");
	//Sort data
	System.out.println("sorting...");
	v.qsort(new ElementComparator() {
	  public boolean preceeds(Object element1, Object element2)
	  {
		return ((String)element1).compareTo((String)element2) < 0;
	  }
	});
	//Output sorted data
	Enumeration e = v.elements();
	while (e.hasMoreElements()) {
	  System.out.println("  " + (String)e.nextElement());
	}
  }

  /**
   * Repeatedly performs both sort and qsort operations on randomly
   * generated arrays. The results of the sorting are compared elementwise.
   * The resulting arrays are also checked for proper ordering.
   */
  public static void extensiveTest()
  {
	final int TESTS = 10;
	final int MAXSIZE = 2000;
	for (int i = 0; i < TESTS; ++i) {
	  int size = (int)(MAXSIZE * Math.random());
	  System.out.print("Testing random vector of size " + size);
	  //Create random vectors
	  System.out.print(" (create)");
	  SortableVector v1, v2;
	  v1 = new SortableVector(size);
	  v2 = new SortableVector(size);
	  for (int j = 0; j < size; ++j) {
		double value = Math.random();
		v1.addElement(new Double(value));
		v2.addElement(new Double(value));
	  }
	  //Sort vectors
	  System.out.print(" (sort)");
	  long tsort = System.currentTimeMillis();
	  v1.sort(new ElementComparator() {
		public boolean preceeds(Object element1, Object element2)
		  {
			double x = ((Double)element1).doubleValue();
			double y = ((Double)element2).doubleValue();
			return x < y;
		  }
	  });
	  tsort = System.currentTimeMillis() - tsort;
	  System.out.print(" (qsort)");
	  long tqsort = System.currentTimeMillis();
	  v2.qsort(new ElementComparator() {
		public boolean preceeds(Object element1, Object element2)
		  {
			double x = ((Double)element1).doubleValue();
			double y = ((Double)element2).doubleValue();
			return x < y;
		  }
	  });
	  tqsort = System.currentTimeMillis() - tqsort;
	  //Compare sorted vectors
	  System.out.println(" (compare)");
	  boolean differing = false;
	  double oldx = -1.0;
	  for (int j = 0; j < size; ++j) {
		double x1 = ((Double)v1.elementAt(j)).doubleValue();
		double x2 = ((Double)v2.elementAt(j)).doubleValue();
		if (x1 != x2) {
		  System.err.println(
			"difference at " + j + ": " + x1 + " != " + x2
		  );
		  differing = true;
		  break;
		}
		if (oldx >= 0 & !(x1 >= oldx)) {
		  System.err.println(
			"elements not in order: " + oldx + " " + x1
		  );
		  differing = true;
		  break;
		}
		oldx = x1;
	  }
	  if (differing) {
		break;
	  }
	  System.out.println(
        "  OK: sort=" + tsort + " qsort=" + tqsort + 
		" (" + Str.getFormatted("%.1f", (double)tsort/(tqsort>0?tqsort:1)) + ":1)"
      );
	}
  }
}
